/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import com.google.protobuf.Message;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.TextFormat;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.discover.RegistrationManager;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.DataFormats;
import org.apache.bookkeeper.versioning.LongVersion;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Cookie {
    private static final Logger LOG = LoggerFactory.getLogger(Cookie.class);
    static final int CURRENT_COOKIE_LAYOUT_VERSION = 4;
    private final int layoutVersion;
    private final String bookieHost;
    private final String journalDirs;
    private final String ledgerDirs;
    private final String instanceId;
    private static final String SEPARATOR = "\t";

    private Cookie(int layoutVersion, String bookieHost, String journalDirs, String ledgerDirs, String instanceId) {
        this.layoutVersion = layoutVersion;
        this.bookieHost = bookieHost;
        this.journalDirs = journalDirs;
        this.ledgerDirs = ledgerDirs;
        this.instanceId = instanceId;
    }

    public static String encodeDirPaths(String[] dirs) {
        StringBuilder b = new StringBuilder();
        b.append(dirs.length);
        for (String d : dirs) {
            b.append(SEPARATOR).append(d);
        }
        return b.toString();
    }

    private static String[] decodeDirPathFromCookie(String s) {
        return s.substring(s.indexOf(SEPARATOR) + SEPARATOR.length()).split(SEPARATOR);
    }

    String[] getLedgerDirPathsFromCookie() {
        return Cookie.decodeDirPathFromCookie(this.ledgerDirs);
    }

    private boolean isSuperSet(String[] superS, String[] subS) {
        HashSet superSet = Sets.newHashSet((Object[])superS);
        HashSet subSet = Sets.newHashSet((Object[])subS);
        return superSet.containsAll(subSet);
    }

    private boolean verifyLedgerDirs(Cookie c, boolean checkIfSuperSet) {
        if (!checkIfSuperSet) {
            return this.ledgerDirs.equals(c.ledgerDirs);
        }
        return this.isSuperSet(Cookie.decodeDirPathFromCookie(this.ledgerDirs), Cookie.decodeDirPathFromCookie(c.ledgerDirs));
    }

    private void verifyInternal(Cookie c, boolean checkIfSuperSet) throws BookieException.InvalidCookieException {
        if (c.layoutVersion < 3 && c.layoutVersion != this.layoutVersion) {
            String errMsg = "Cookie is of too old version " + c.layoutVersion;
            LOG.error(errMsg);
            throw new BookieException.InvalidCookieException(errMsg);
        }
        if (!(c.layoutVersion >= 3 && c.bookieHost.equals(this.bookieHost) && c.journalDirs.equals(this.journalDirs) && this.verifyLedgerDirs(c, checkIfSuperSet))) {
            String errMsg = "Cookie [" + this + "] is not matching with [" + c + "]";
            throw new BookieException.InvalidCookieException(errMsg);
        }
        if (this.instanceId == null && c.instanceId != null || this.instanceId != null && !this.instanceId.equals(c.instanceId)) {
            String errMsg = "instanceId " + this.instanceId + " is not matching with " + c.instanceId;
            throw new BookieException.InvalidCookieException(errMsg);
        }
    }

    public void verify(Cookie c) throws BookieException.InvalidCookieException {
        this.verifyInternal(c, false);
    }

    public void verifyIsSuperSet(Cookie c) throws BookieException.InvalidCookieException {
        this.verifyInternal(c, true);
    }

    public String toString() {
        if (this.layoutVersion <= 3) {
            return this.toStringVersion3();
        }
        DataFormats.CookieFormat.Builder builder = DataFormats.CookieFormat.newBuilder();
        builder.setBookieHost(this.bookieHost);
        builder.setJournalDir(this.journalDirs);
        builder.setLedgerDirs(this.ledgerDirs);
        if (null != this.instanceId) {
            builder.setInstanceId(this.instanceId);
        }
        StringBuilder b = new StringBuilder();
        b.append(4).append("\n");
        b.append(TextFormat.printToString((MessageOrBuilder)builder.build()));
        return b.toString();
    }

    private String toStringVersion3() {
        StringBuilder b = new StringBuilder();
        b.append(4).append("\n").append(this.bookieHost).append("\n").append(this.journalDirs).append("\n").append(this.ledgerDirs).append("\n");
        return b.toString();
    }

    private static Builder parse(BufferedReader reader) throws IOException {
        Builder cBuilder = Cookie.newBuilder();
        int layoutVersion = 0;
        String line = reader.readLine();
        if (null == line) {
            throw new EOFException("Exception in parsing cookie");
        }
        try {
            layoutVersion = Integer.parseInt(line.trim());
            cBuilder.setLayoutVersion(layoutVersion);
        }
        catch (NumberFormatException e) {
            throw new IOException("Invalid string '" + line.trim() + "', cannot parse cookie.");
        }
        if (layoutVersion == 3) {
            cBuilder.setBookieHost(reader.readLine());
            cBuilder.setJournalDirs(reader.readLine());
            cBuilder.setLedgerDirs(reader.readLine());
        } else if (layoutVersion >= 4) {
            DataFormats.CookieFormat.Builder cfBuilder = DataFormats.CookieFormat.newBuilder();
            TextFormat.merge((Readable)reader, (Message.Builder)cfBuilder);
            DataFormats.CookieFormat data = cfBuilder.build();
            cBuilder.setBookieHost(data.getBookieHost());
            cBuilder.setJournalDirs(data.getJournalDir());
            cBuilder.setLedgerDirs(data.getLedgerDirs());
            if (null != data.getInstanceId() && !data.getInstanceId().isEmpty()) {
                cBuilder.setInstanceId(data.getInstanceId());
            }
        }
        return cBuilder;
    }

    public void writeToDirectory(File directory) throws IOException {
        File versionFile = new File(directory, "VERSION");
        this.writeToFile(versionFile);
    }

    public void writeToFile(File versionFile) throws IOException {
        try (FileOutputStream fos = new FileOutputStream(versionFile);
             BufferedWriter bw = new BufferedWriter(new OutputStreamWriter((OutputStream)fos, StandardCharsets.UTF_8));){
            bw.write(this.toString());
        }
    }

    public void writeToRegistrationManager(RegistrationManager rm, ServerConfiguration conf, Version version) throws BookieException {
        BookieSocketAddress address = null;
        try {
            address = Bookie.getBookieAddress(conf);
        }
        catch (UnknownHostException e) {
            throw new BookieException.UnknownBookieIdException(e);
        }
        byte[] data = this.toString().getBytes(StandardCharsets.UTF_8);
        rm.writeCookie(address.toString(), new Versioned<byte[]>(data, version));
    }

    public void deleteFromRegistrationManager(RegistrationManager rm, ServerConfiguration conf, Version version) throws BookieException {
        BookieSocketAddress address = null;
        try {
            address = Bookie.getBookieAddress(conf);
        }
        catch (UnknownHostException e) {
            throw new BookieException.UnknownBookieIdException(e);
        }
        this.deleteFromRegistrationManager(rm, address, version);
    }

    public void deleteFromRegistrationManager(RegistrationManager rm, BookieSocketAddress address, Version version) throws BookieException {
        if (!(version instanceof LongVersion)) {
            throw new IllegalArgumentException("Invalid version type, expected ZkVersion type");
        }
        rm.removeCookie(address.toString(), version);
    }

    static Builder generateCookie(ServerConfiguration conf) throws UnknownHostException {
        Builder builder = Cookie.newBuilder();
        builder.setLayoutVersion(4);
        builder.setBookieHost(Bookie.getBookieAddress(conf).toString());
        builder.setJournalDirs(Joiner.on((char)',').join((Object[])conf.getJournalDirNames()));
        builder.setLedgerDirs(Cookie.encodeDirPaths(conf.getLedgerDirNames()));
        return builder;
    }

    public static Versioned<Cookie> readFromRegistrationManager(RegistrationManager rm, ServerConfiguration conf) throws BookieException {
        try {
            return Cookie.readFromRegistrationManager(rm, Bookie.getBookieAddress(conf));
        }
        catch (UnknownHostException e) {
            throw new BookieException.UnknownBookieIdException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Versioned<Cookie> readFromRegistrationManager(RegistrationManager rm, BookieSocketAddress address) throws BookieException {
        Versioned<byte[]> cookieData = rm.readCookie(address.toString());
        try (BufferedReader reader = new BufferedReader(new StringReader(new String(cookieData.getValue(), StandardCharsets.UTF_8)));){
            Builder builder = Cookie.parse(reader);
            Cookie cookie = builder.build();
            Versioned<Cookie> versioned = new Versioned<Cookie>(cookie, cookieData.getVersion());
            return versioned;
        }
        catch (IOException ioe) {
            throw new BookieException.InvalidCookieException(ioe);
        }
    }

    public static Cookie readFromDirectory(File directory) throws IOException {
        File versionFile = new File(directory, "VERSION");
        try (BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(versionFile), StandardCharsets.UTF_8));){
            Cookie cookie = Cookie.parse(reader).build();
            return cookie;
        }
    }

    public boolean isBookieHostCreatedFromIp() throws IOException {
        int port;
        String[] parts = this.bookieHost.split(":");
        if (parts.length != 2) {
            throw new IOException(this.bookieHost + " does not have the form host:port");
        }
        try {
            port = Integer.parseInt(parts[1]);
        }
        catch (NumberFormatException e) {
            throw new IOException(this.bookieHost + " does not have the form host:port");
        }
        InetSocketAddress addr = new InetSocketAddress(parts[0], port);
        return addr.toString().startsWith("/");
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(Cookie oldCookie) {
        return new Builder(oldCookie.layoutVersion, oldCookie.bookieHost, oldCookie.journalDirs, oldCookie.ledgerDirs, oldCookie.instanceId);
    }

    public static class Builder {
        private int layoutVersion = 4;
        private String bookieHost = null;
        private String journalDirs = null;
        private String ledgerDirs = null;
        private String instanceId = null;

        private Builder() {
        }

        private Builder(int layoutVersion, String bookieHost, String journalDirs, String ledgerDirs, String instanceId) {
            this.layoutVersion = layoutVersion;
            this.bookieHost = bookieHost;
            this.journalDirs = journalDirs;
            this.ledgerDirs = ledgerDirs;
            this.instanceId = instanceId;
        }

        public Builder setLayoutVersion(int layoutVersion) {
            this.layoutVersion = layoutVersion;
            return this;
        }

        public Builder setBookieHost(String bookieHost) {
            this.bookieHost = bookieHost;
            return this;
        }

        public Builder setJournalDirs(String journalDirs) {
            this.journalDirs = journalDirs;
            return this;
        }

        public Builder setLedgerDirs(String ledgerDirs) {
            this.ledgerDirs = ledgerDirs;
            return this;
        }

        public Builder setInstanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public Cookie build() {
            return new Cookie(this.layoutVersion, this.bookieHost, this.journalDirs, this.ledgerDirs, this.instanceId);
        }
    }
}

