/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.io;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HexFormat;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import org.eclipse.jetty.io.AbstractEndPoint;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.FillInterest;
import org.eclipse.jetty.io.WriteFlusher;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.thread.AutoLock;
import org.eclipse.jetty.util.thread.Invocable;
import org.eclipse.jetty.util.thread.Scheduler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryEndPointPipe
implements EndPoint.Pipe {
    private final LocalEndPoint localEndPoint;
    private final RemoteEndPoint remoteEndPoint;
    private final Consumer<Invocable.Task> taskConsumer;

    public MemoryEndPointPipe(Scheduler scheduler, Consumer<Invocable.Task> consumer, SocketAddress socketAddress) {
        this.localEndPoint = new LocalEndPoint(scheduler, socketAddress);
        this.remoteEndPoint = new RemoteEndPoint(scheduler, new MemorySocketAddress());
        this.localEndPoint.setPeerEndPoint(this.remoteEndPoint);
        this.remoteEndPoint.setPeerEndPoint(this.localEndPoint);
        this.taskConsumer = consumer;
    }

    @Override
    public EndPoint getLocalEndPoint() {
        return this.localEndPoint;
    }

    @Override
    public EndPoint getRemoteEndPoint() {
        return this.remoteEndPoint;
    }

    public void setLocalEndPointMaxCapacity(int maxCapacity) {
        this.localEndPoint.setMaxCapacity(maxCapacity);
    }

    public void setRemoteEndPointMaxCapacity(int maxCapacity) {
        this.remoteEndPoint.setMaxCapacity(maxCapacity);
    }

    private class LocalEndPoint
    extends MemoryEndPoint {
        private LocalEndPoint(Scheduler scheduler, SocketAddress socketAddress) {
            super(scheduler, socketAddress);
        }
    }

    private class RemoteEndPoint
    extends MemoryEndPoint {
        private RemoteEndPoint(Scheduler scheduler, SocketAddress socketAddress) {
            super(scheduler, socketAddress);
        }
    }

    private static class MemorySocketAddress
    extends SocketAddress {
        private static final AtomicLong ID = new AtomicLong();
        private final long id = ID.incrementAndGet();
        private final String address = "[memory:/%s]".formatted(HexFormat.of().formatHex(ByteBuffer.allocate(8).putLong(this.id).array()));

        private MemorySocketAddress() {
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof MemorySocketAddress) {
                MemorySocketAddress that = (MemorySocketAddress)obj;
                return this.id == that.id;
            }
            return false;
        }

        public int hashCode() {
            return Objects.hash(this.id);
        }

        public String toString() {
            return this.address;
        }
    }

    private class MemoryEndPoint
    extends AbstractEndPoint {
        private static final Logger LOG = LoggerFactory.getLogger(MemoryEndPoint.class);
        private static final ByteBuffer EOF = ByteBuffer.allocate(0);
        private final AutoLock lock;
        private final Deque<ByteBuffer> byteBuffers;
        private final SocketAddress localAddress;
        private MemoryEndPoint peerEndPoint;
        private Invocable.Task fillableTask;
        private Invocable.Task completeWriteTask;
        private long maxCapacity;
        private long capacity;

        private MemoryEndPoint(Scheduler scheduler, SocketAddress localAddress) {
            super(scheduler);
            this.lock = new AutoLock();
            this.byteBuffers = new ArrayDeque<ByteBuffer>();
            this.localAddress = localAddress;
        }

        void setPeerEndPoint(MemoryEndPoint peerEndPoint) {
            this.peerEndPoint = peerEndPoint;
            this.fillableTask = new FillableTask(peerEndPoint.getFillInterest());
            this.completeWriteTask = new CompleteWriteTask(peerEndPoint.getWriteFlusher());
        }

        public long getMaxCapacity() {
            return this.maxCapacity;
        }

        public void setMaxCapacity(long maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        @Override
        public Object getTransport() {
            return null;
        }

        @Override
        public SocketAddress getLocalSocketAddress() {
            return this.localAddress;
        }

        @Override
        public SocketAddress getRemoteSocketAddress() {
            return this.peerEndPoint.getLocalSocketAddress();
        }

        @Override
        protected void onIncompleteFlush() {
        }

        @Override
        protected void needsFillInterest() {
        }

        @Override
        public int fill(ByteBuffer buffer) throws IOException {
            if (!this.isOpen()) {
                throw new IOException("closed");
            }
            if (this.isInputShutdown()) {
                return -1;
            }
            int filled = this.peerEndPoint.fillInto(buffer);
            if (LOG.isDebugEnabled()) {
                LOG.debug("filled {} from {}", (Object)filled, (Object)this);
            }
            if (filled > 0) {
                this.notIdle();
                this.onFilled();
            } else if (filled < 0) {
                this.shutdownInput();
            }
            return filled;
        }

        private int fillInto(ByteBuffer buffer) {
            int filled = 0;
            try (AutoLock ignored = this.lock.lock();){
                while (true) {
                    int n;
                    ByteBuffer data;
                    if ((data = this.byteBuffers.peek()) == null) {
                        n = filled;
                        return n;
                    }
                    if (data == EOF) {
                        n = filled > 0 ? filled : -1;
                        return n;
                    }
                    int length = data.remaining();
                    int copied = BufferUtil.append(buffer, data);
                    this.capacity -= (long)copied;
                    filled += copied;
                    if (copied < length) {
                        int n2 = filled;
                        return n2;
                    }
                    this.byteBuffers.poll();
                }
            }
        }

        private void onFilled() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("filled, notifying completeWrite {}", (Object)this);
            }
            MemoryEndPointPipe.this.taskConsumer.accept(this.completeWriteTask);
        }

        @Override
        public void fillInterested(Callback callback) {
            try (AutoLock ignored = this.lock.lock();){
                if (this.peerEndPoint.byteBuffers.isEmpty()) {
                    super.fillInterested(callback);
                    return;
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("fill interested, data available {}", (Object)this);
            }
            callback.succeeded();
        }

        @Override
        public boolean tryFillInterested(Callback callback) {
            try (AutoLock ignored = this.lock.lock();){
                if (this.peerEndPoint.byteBuffers.isEmpty()) {
                    boolean bl = super.tryFillInterested(callback);
                    return bl;
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("try fill interested, data available {}", (Object)this);
            }
            callback.succeeded();
            return false;
        }

        @Override
        public boolean flush(ByteBuffer ... buffers) throws IOException {
            if (!this.isOpen()) {
                throw new IOException("closed");
            }
            if (this.isOutputShutdown()) {
                throw new IOException("shutdown");
            }
            long flushed = 0L;
            boolean result = true;
            try (AutoLock ignored = this.lock.lock();){
                for (ByteBuffer buffer : buffers) {
                    int remaining = buffer.remaining();
                    if (remaining == 0) continue;
                    ByteBuffer copy = this.lockedCopy(buffer);
                    if (copy == null) {
                        result = false;
                    } else {
                        this.byteBuffers.offer(copy);
                        int length = copy.remaining();
                        this.capacity += (long)length;
                        flushed += (long)length;
                        if (length >= remaining) continue;
                        result = false;
                    }
                    break;
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("flushed {} to {}", (Object)flushed, (Object)this);
            }
            if (flushed > 0L) {
                this.notIdle();
                this.onFlushed();
            }
            return result;
        }

        private ByteBuffer lockedCopy(ByteBuffer buffer) {
            int length = buffer.remaining();
            long maxCapacity = this.getMaxCapacity();
            if (maxCapacity > 0L) {
                long space = maxCapacity - this.capacity;
                if (space == 0L) {
                    return null;
                }
                length = (int)Math.min((long)length, space);
            }
            ByteBuffer copy = buffer.isDirect() ? ByteBuffer.allocateDirect(length) : ByteBuffer.allocate(length);
            copy.put(0, buffer, buffer.position(), length);
            buffer.position(buffer.position() + length);
            return copy;
        }

        @Override
        protected void doShutdownOutput() {
            super.doShutdownOutput();
            try (AutoLock ignored = this.lock.lock();){
                this.byteBuffers.offer(EOF);
            }
            this.onFlushed();
        }

        @Override
        protected void doClose() {
            super.doClose();
            try (AutoLock ignored = this.lock.lock();){
                ByteBuffer last = this.byteBuffers.peekLast();
                if (last != EOF) {
                    this.byteBuffers.offer(EOF);
                }
            }
            this.onFlushed();
        }

        private void onFlushed() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("flushed, notifying fillable {}", (Object)this);
            }
            MemoryEndPointPipe.this.taskConsumer.accept(this.fillableTask);
        }
    }

    private record CompleteWriteTask(WriteFlusher writeFlusher) implements Invocable.Task
    {
        @Override
        public void run() {
            this.writeFlusher.completeWrite();
        }

        @Override
        public Invocable.InvocationType getInvocationType() {
            return this.writeFlusher.getCallbackInvocationType();
        }
    }

    private record FillableTask(FillInterest fillInterest) implements Invocable.Task
    {
        @Override
        public void run() {
            this.fillInterest.fillable();
        }

        @Override
        public Invocable.InvocationType getInvocationType() {
            return this.fillInterest.getCallbackInvocationType();
        }
    }
}

