/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.gateway.service.operation;

import java.lang.reflect.Field;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.FutureTask;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.time.Deadline;
import org.apache.flink.shaded.guava31.com.google.common.util.concurrent.Uninterruptibles;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.gateway.api.operation.OperationHandle;
import org.apache.flink.table.gateway.api.operation.OperationStatus;
import org.apache.flink.table.gateway.api.results.FetchOrientation;
import org.apache.flink.table.gateway.api.results.OperationInfo;
import org.apache.flink.table.gateway.api.results.ResultSet;
import org.apache.flink.table.gateway.api.utils.SqlGatewayException;
import org.apache.flink.table.gateway.service.result.NotReadyResult;
import org.apache.flink.table.gateway.service.result.ResultFetcher;
import org.apache.flink.table.gateway.service.utils.SqlCancelException;
import org.apache.flink.table.gateway.service.utils.SqlExecutionException;
import org.apache.flink.util.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class OperationManager {
    private static final Logger LOG = LoggerFactory.getLogger(OperationManager.class);
    private final ReadWriteLock stateLock = new ReentrantReadWriteLock();
    private final Map<OperationHandle, Operation> submittedOperations;
    private final ExecutorService service;
    private final Semaphore operationLock;
    private boolean isRunning;

    public OperationManager(ExecutorService service) {
        this.service = service;
        this.submittedOperations = new HashMap<OperationHandle, Operation>();
        this.operationLock = new Semaphore(1);
        this.isRunning = true;
    }

    public OperationHandle submitOperation(Callable<ResultSet> executor) {
        OperationHandle handle = OperationHandle.create();
        Operation operation = new Operation(handle, () -> {
            ResultSet resultSet = (ResultSet)executor.call();
            return ResultFetcher.fromResults(handle, resultSet.getResultSchema(), resultSet.getData());
        });
        this.submitOperationInternal(handle, operation);
        return handle;
    }

    public OperationHandle submitOperation(Function<OperationHandle, ResultFetcher> fetcherSupplier) {
        OperationHandle handle = OperationHandle.create();
        Operation operation = new Operation(handle, () -> (ResultFetcher)fetcherSupplier.apply(handle));
        this.submitOperationInternal(handle, operation);
        return handle;
    }

    public void cancelOperation(OperationHandle operationHandle) {
        this.getOperation(operationHandle).cancel();
    }

    public void closeOperation(OperationHandle operationHandle) {
        this.writeLock(() -> {
            Operation opToRemove = this.submittedOperations.remove(operationHandle);
            if (opToRemove != null) {
                opToRemove.close();
            }
        });
    }

    public void awaitOperationTermination(OperationHandle operationHandle) throws Exception {
        this.getOperation(operationHandle).awaitTermination();
    }

    public OperationInfo getOperationInfo(OperationHandle operationHandle) {
        return this.getOperation(operationHandle).getOperationInfo();
    }

    public ResolvedSchema getOperationResultSchema(OperationHandle operationHandle) throws Exception {
        return this.getOperation(operationHandle).getResultSchema();
    }

    public ResultSet fetchResults(OperationHandle operationHandle, long token, int maxRows) {
        return this.getOperation(operationHandle).fetchResults(token, maxRows);
    }

    public ResultSet fetchResults(OperationHandle operationHandle, FetchOrientation orientation, int maxRows) {
        return this.getOperation(operationHandle).fetchResults(orientation, maxRows);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        this.stateLock.writeLock().lock();
        Exception closeException = null;
        try {
            this.isRunning = false;
            IOUtils.closeAll(this.submittedOperations.values(), Throwable.class);
        }
        catch (Exception e) {
            closeException = e;
        }
        finally {
            this.submittedOperations.clear();
            this.stateLock.writeLock().unlock();
        }
        try {
            this.operationLock.acquire();
        }
        catch (Exception e) {
            LOG.error("Failed to wait all operation closed.", (Throwable)e);
        }
        finally {
            this.operationLock.release();
        }
        LOG.debug("Closes the Operation Manager.");
        if (closeException != null) {
            throw new SqlExecutionException("Failed to close the OperationManager.", closeException);
        }
    }

    @VisibleForTesting
    public int getOperationCount() {
        return this.submittedOperations.size();
    }

    @VisibleForTesting
    public Operation getOperation(OperationHandle operationHandle) {
        return this.readLock(() -> {
            Operation operation = this.submittedOperations.get(operationHandle);
            if (operation == null) {
                throw new SqlGatewayException(String.format("Can not find the submitted operation in the OperationManager with the %s.", operationHandle));
            }
            return operation;
        });
    }

    private void submitOperationInternal(OperationHandle handle, Operation operation) {
        this.writeLock(() -> this.submittedOperations.put(handle, operation));
        operation.run();
    }

    private void writeLock(Runnable runner) {
        this.stateLock.writeLock().lock();
        try {
            if (!this.isRunning) {
                throw new SqlGatewayException("The OperationManager is closed.");
            }
            runner.run();
        }
        finally {
            this.stateLock.writeLock().unlock();
        }
    }

    private <T> T readLock(Supplier<T> supplier) {
        this.stateLock.readLock().lock();
        try {
            if (!this.isRunning) {
                throw new SqlGatewayException("The OperationManager is closed.");
            }
            T t = supplier.get();
            return t;
        }
        finally {
            this.stateLock.readLock().unlock();
        }
    }

    @VisibleForTesting
    public class Operation
    implements AutoCloseable {
        private static final long WAIT_CLEAN_UP_MILLISECONDS = 5000L;
        private final OperationHandle operationHandle;
        private final AtomicReference<OperationStatus> status;
        private final Callable<ResultFetcher> resultSupplier;
        private volatile FutureTask<?> invocation;
        private volatile ResultFetcher resultFetcher;
        private volatile SqlExecutionException operationError;

        public Operation(OperationHandle operationHandle, Callable<ResultFetcher> resultSupplier) {
            this.operationHandle = operationHandle;
            this.status = new AtomicReference<OperationStatus>(OperationStatus.INITIALIZED);
            this.resultSupplier = resultSupplier;
        }

        void runBefore() {
            this.updateState(OperationStatus.RUNNING);
        }

        void runAfter() {
            this.updateState(OperationStatus.FINISHED);
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        public void run() {
            try {
                OperationManager.this.operationLock.acquire();
                LOG.debug(String.format("Operation %s acquires the operation lock.", this.operationHandle));
                this.updateState(OperationStatus.PENDING);
                Runnable work = () -> {
                    try {
                        this.runBefore();
                        this.resultFetcher = this.resultSupplier.call();
                        this.runAfter();
                    }
                    catch (InterruptedException e) {
                        LOG.error(String.format("Operation %s is interrupted.", this.operationHandle), (Throwable)e);
                    }
                    catch (Throwable t) {
                        this.processThrowable(t);
                    }
                };
                FutureTask<Void> copiedTask = new FutureTask<Void>(work, null){

                    @Override
                    protected void done() {
                        LOG.debug(String.format("Release the operation lock: %s when task completes.", Operation.this.operationHandle));
                        OperationManager.this.operationLock.release();
                    }
                };
                OperationManager.this.service.submit(copiedTask);
                this.invocation = copiedTask;
                OperationStatus current = this.status.get();
                if (current == OperationStatus.CLOSED || current == OperationStatus.CANCELED) {
                    LOG.debug(String.format("The current status is %s after updating the operation %s status to %s. Close the resources.", new Object[]{current, this.operationHandle, OperationStatus.PENDING}));
                    this.closeResources();
                }
                if (this.invocation != null) return;
            }
            catch (Throwable t) {
                try {
                    this.processThrowable(t);
                    throw new SqlGatewayException("Failed to submit the operation to the thread pool.", t);
                }
                catch (Throwable throwable) {
                    if (this.invocation != null) throw throwable;
                    LOG.debug(String.format("Operation %s releases the operation lock when failed to submit the operation to the pool.", this.operationHandle));
                    OperationManager.this.operationLock.release();
                    throw throwable;
                }
            }
            LOG.debug(String.format("Operation %s releases the operation lock when failed to submit the operation to the pool.", this.operationHandle));
            OperationManager.this.operationLock.release();
            return;
        }

        public void cancel() {
            this.updateState(OperationStatus.CANCELED);
            this.closeResources();
        }

        @Override
        public void close() {
            this.updateState(OperationStatus.CLOSED);
            this.closeResources();
        }

        public ResultSet fetchResults(long token, int maxRows) {
            return this.fetchResultsInternal(() -> this.resultFetcher.fetchResults(token, maxRows));
        }

        public ResultSet fetchResults(FetchOrientation orientation, int maxRows) {
            return this.fetchResultsInternal(() -> this.resultFetcher.fetchResults(orientation, maxRows));
        }

        public ResolvedSchema getResultSchema() throws Exception {
            this.awaitTermination();
            OperationStatus current = this.status.get();
            if (current != OperationStatus.FINISHED) {
                throw new IllegalStateException(String.format("The result schema is available when the Operation is in FINISHED state but the current status is %s.", this.status));
            }
            return this.resultFetcher.getResultSchema();
        }

        public OperationInfo getOperationInfo() {
            return new OperationInfo(this.status.get(), this.operationError);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void awaitTermination() throws Exception {
            AtomicReference<OperationStatus> atomicReference = this.status;
            synchronized (atomicReference) {
                while (!this.status.get().isTerminalStatus()) {
                    this.status.wait();
                }
            }
            OperationStatus current = this.status.get();
            if (current == OperationStatus.ERROR) {
                throw this.operationError;
            }
        }

        private ResultSet fetchResultsInternal(Supplier<ResultSet> results) {
            OperationStatus currentStatus = this.status.get();
            if (currentStatus == OperationStatus.ERROR) {
                throw this.operationError;
            }
            if (currentStatus == OperationStatus.FINISHED) {
                return results.get();
            }
            if (currentStatus == OperationStatus.RUNNING || currentStatus == OperationStatus.PENDING || currentStatus == OperationStatus.INITIALIZED) {
                return NotReadyResult.INSTANCE;
            }
            throw new SqlGatewayException(String.format("Can not fetch results from the %s in %s status.", new Object[]{this.operationHandle, currentStatus}));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void updateState(OperationStatus toStatus) {
            OperationStatus currentStatus;
            do {
                boolean isValid;
                if (isValid = OperationStatus.isValidStatusTransition(currentStatus = this.status.get(), toStatus)) continue;
                String message = String.format("Failed to convert the Operation Status from %s to %s for %s.", new Object[]{currentStatus, toStatus, this.operationHandle});
                throw new SqlGatewayException(message);
            } while (!this.status.compareAndSet(currentStatus, toStatus));
            AtomicReference<OperationStatus> atomicReference = this.status;
            synchronized (atomicReference) {
                this.status.notifyAll();
            }
            LOG.debug(String.format("Convert operation %s from %s to %s.", new Object[]{this.operationHandle, currentStatus, toStatus}));
        }

        private void closeResources() {
            if (this.invocation != null && !this.invocation.isDone()) {
                this.invocation.cancel(true);
                this.waitTaskCleanup(this.invocation);
                LOG.debug(String.format("Cancel the operation %s.", this.operationHandle));
            }
            if (this.resultFetcher != null) {
                this.resultFetcher.close();
            }
        }

        private void processThrowable(Throwable t) {
            String msg = String.format("Failed to execute the operation %s.", this.operationHandle);
            LOG.error(msg, t);
            this.operationError = new SqlExecutionException(msg, t);
            this.updateState(OperationStatus.ERROR);
        }

        private void waitTaskCleanup(FutureTask<?> invocation) {
            Optional<Thread> threadOptional;
            Deadline deadline = Deadline.fromNow((Duration)Duration.ofMillis(5000L));
            while (deadline.hasTimeLeft()) {
                threadOptional = this.getThreadInFuture(invocation);
                if (!threadOptional.isPresent()) {
                    return;
                }
                Uninterruptibles.sleepUninterruptibly((long)1L, (TimeUnit)TimeUnit.MILLISECONDS);
            }
            threadOptional = this.getThreadInFuture(invocation);
            threadOptional.ifPresent(this::throwExceptionWithThreadStackTrace);
        }

        private Optional<Thread> getThreadInFuture(FutureTask<?> invocation) {
            try {
                Class<FutureTask> k = FutureTask.class;
                Field runnerField = k.getDeclaredField("runner");
                runnerField.setAccessible(true);
                Thread t = (Thread)runnerField.get(invocation);
                return Optional.of(t);
            }
            catch (Throwable e) {
                return Optional.empty();
            }
        }

        private void throwExceptionWithThreadStackTrace(Thread thread) {
            StackTraceElement[] stack = thread.getStackTrace();
            StringBuilder stackTraceStr = new StringBuilder();
            for (StackTraceElement e : stack) {
                stackTraceStr.append("\tat ").append(e).append("\n");
            }
            String msg = String.format("Operation '%s' did not react to \"Future.cancel(true)\" and is stuck for %s seconds in method.\nThread name: %s, thread state: %s, thread stacktrace:\n%s", new Object[]{this.operationHandle, 5L, thread.getName(), thread.getState(), stackTraceStr});
            throw new SqlCancelException(msg);
        }
    }
}

