/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ranger.biz;

import org.apache.ranger.common.MessageEnums;
import org.apache.ranger.common.RESTErrorUtil;
import org.apache.ranger.common.RangerConstants;
import org.apache.ranger.db.RangerDaoManager;
import org.apache.ranger.db.XXGroupDao;
import org.apache.ranger.db.XXPolicyDao;
import org.apache.ranger.db.XXResourceDefDao;
import org.apache.ranger.db.XXRoleDao;
import org.apache.ranger.db.XXSecurityZoneRefGroupDao;
import org.apache.ranger.db.XXSecurityZoneRefResourceDao;
import org.apache.ranger.db.XXSecurityZoneRefRoleDao;
import org.apache.ranger.db.XXSecurityZoneRefServiceDao;
import org.apache.ranger.db.XXSecurityZoneRefTagServiceDao;
import org.apache.ranger.db.XXSecurityZoneRefUserDao;
import org.apache.ranger.db.XXServiceDao;
import org.apache.ranger.db.XXServiceDefDao;
import org.apache.ranger.db.XXUserDao;
import org.apache.ranger.entity.XXGroup;
import org.apache.ranger.entity.XXPolicy;
import org.apache.ranger.entity.XXResourceDef;
import org.apache.ranger.entity.XXRole;
import org.apache.ranger.entity.XXSecurityZoneRefGroup;
import org.apache.ranger.entity.XXSecurityZoneRefResource;
import org.apache.ranger.entity.XXSecurityZoneRefRole;
import org.apache.ranger.entity.XXSecurityZoneRefService;
import org.apache.ranger.entity.XXSecurityZoneRefTagService;
import org.apache.ranger.entity.XXSecurityZoneRefUser;
import org.apache.ranger.entity.XXService;
import org.apache.ranger.entity.XXServiceDef;
import org.apache.ranger.entity.XXUser;
import org.apache.ranger.plugin.model.RangerPolicy;
import org.apache.ranger.plugin.model.RangerSecurityZone;
import org.apache.ranger.plugin.model.RangerService;
import org.apache.ranger.service.RangerPolicyService;
import org.apache.ranger.service.RangerServiceService;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.junit.jupiter.MockitoExtension;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

/**
 * @generated by Cursor
 * @description <Unit Test for TestSecurityZoneRefUpdater class>
 */
@ExtendWith(MockitoExtension.class)
@TestMethodOrder(MethodOrderer.MethodName.class)
public class TestSecurityZoneRefUpdater {
    private static void setField(Object target, Class<?> declaring, String fieldName, Object value) throws Exception {
        Field f = declaring.getDeclaredField(fieldName);
        f.setAccessible(true);
        f.set(target, value);
    }

    @Test
    public void test01_cleanupRefTables_handlesNullAndDeletesAll() throws Exception {
        SecurityZoneRefUpdater up = new SecurityZoneRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        XXSecurityZoneRefServiceDao sd = mock(XXSecurityZoneRefServiceDao.class);
        XXSecurityZoneRefTagServiceDao td = mock(XXSecurityZoneRefTagServiceDao.class);
        XXSecurityZoneRefResourceDao rd = mock(XXSecurityZoneRefResourceDao.class);
        XXSecurityZoneRefUserDao ud = mock(XXSecurityZoneRefUserDao.class);
        XXSecurityZoneRefGroupDao gd = mock(XXSecurityZoneRefGroupDao.class);
        XXSecurityZoneRefRoleDao rld = mock(XXSecurityZoneRefRoleDao.class);
        when(dao.getXXSecurityZoneRefService()).thenReturn(sd);
        when(dao.getXXSecurityZoneRefTagService()).thenReturn(td);
        when(dao.getXXSecurityZoneRefResource()).thenReturn(rd);
        when(dao.getXXSecurityZoneRefUser()).thenReturn(ud);
        when(dao.getXXSecurityZoneRefGroup()).thenReturn(gd);
        when(dao.getXXSecurityZoneRefRole()).thenReturn(rld);
        setField(up, SecurityZoneRefUpdater.class, "daoMgr", dao);

        RangerSecurityZone nullZone = new RangerSecurityZone();
        nullZone.setId(null);
        Assertions.assertFalse(up.cleanupRefTables(nullZone));

        RangerSecurityZone zone = new RangerSecurityZone();
        zone.setId(5L);
        when(sd.findByZoneId(5L)).thenReturn(Collections.singletonList(new XXSecurityZoneRefService()));
        when(td.findByZoneId(5L)).thenReturn(Collections.singletonList(new XXSecurityZoneRefTagService()));
        when(rd.findByZoneId(5L)).thenReturn(Collections.singletonList(new XXSecurityZoneRefResource()));
        when(ud.findByZoneId(5L)).thenReturn(Collections.singletonList(new XXSecurityZoneRefUser()));
        when(gd.findByZoneId(5L)).thenReturn(Collections.singletonList(new XXSecurityZoneRefGroup()));
        when(rld.findByZoneId(5L)).thenReturn(Collections.singletonList(new XXSecurityZoneRefRole()));
        Assertions.assertTrue(up.cleanupRefTables(zone));
        verify(sd, times(1)).remove(any(XXSecurityZoneRefService.class));
        verify(td, times(1)).remove(any(XXSecurityZoneRefTagService.class));
        verify(rd, times(1)).remove(any(XXSecurityZoneRefResource.class));
        verify(ud, times(1)).remove(any(XXSecurityZoneRefUser.class));
        verify(gd, times(1)).remove(any(XXSecurityZoneRefGroup.class));
        verify(rld, times(1)).remove(any(XXSecurityZoneRefRole.class));
    }

    @Test
    public void test02_createNewZoneMapping_happyPath_createsAllRefs() throws Exception {
        SecurityZoneRefUpdater up = new SecurityZoneRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        RangerServiceService svcService = mock(RangerServiceService.class);
        RESTErrorUtil rest = mock(RESTErrorUtil.class);
        setField(up, SecurityZoneRefUpdater.class, "daoMgr", dao);
        setField(up, SecurityZoneRefUpdater.class, "svcService", svcService);
        setField(up, SecurityZoneRefUpdater.class, "restErrorUtil", rest);

        // zone-ref DAOs for create() verifications
        XXSecurityZoneRefServiceDao sd = mock(XXSecurityZoneRefServiceDao.class);
        XXSecurityZoneRefTagServiceDao td = mock(XXSecurityZoneRefTagServiceDao.class);
        XXSecurityZoneRefResourceDao rd = mock(XXSecurityZoneRefResourceDao.class);
        XXSecurityZoneRefUserDao ud = mock(XXSecurityZoneRefUserDao.class);
        XXSecurityZoneRefGroupDao gd = mock(XXSecurityZoneRefGroupDao.class);
        XXSecurityZoneRefRoleDao rld = mock(XXSecurityZoneRefRoleDao.class);
        when(dao.getXXSecurityZoneRefService()).thenReturn(sd);
        when(dao.getXXSecurityZoneRefTagService()).thenReturn(td);
        when(dao.getXXSecurityZoneRefResource()).thenReturn(rd);
        when(dao.getXXSecurityZoneRefUser()).thenReturn(ud);
        when(dao.getXXSecurityZoneRefGroup()).thenReturn(gd);
        when(dao.getXXSecurityZoneRefRole()).thenReturn(rld);

        XXService xsvc = new XXService();
        xsvc.setId(10L);
        xsvc.setName("s");
        xsvc.setType(Long.valueOf(RangerConstants.TAG_SERVICE_TYPE));
        XXServiceDef xsd = new XXServiceDef();
        xsd.setId(100L);
        xsd.setName("hdfs");
        XXResourceDef xrd = new XXResourceDef();
        xrd.setId(1000L);
        xrd.setName("path");
        XXServiceDefDao xServiceDefDao = mock(XXServiceDefDao.class);
        XXResourceDefDao xResourceDefDao = mock(XXResourceDefDao.class);
        XXServiceDao xServiceDao = mock(XXServiceDao.class);
        when(dao.getXXService()).thenReturn(xServiceDao);
        when(dao.getXXServiceDef()).thenReturn(xServiceDefDao);
        when(dao.getXXResourceDef()).thenReturn(xResourceDefDao);
        when(xServiceDao.findByName("s")).thenReturn(xsvc);
        RangerService rs = new RangerService();
        rs.setType("hdfs");
        when(svcService.getPopulatedViewObject(xsvc)).thenReturn(rs);
        when(xServiceDefDao.findByName("hdfs")).thenReturn(xsd);
        when(xResourceDefDao.findByNameAndServiceDefId("path", 100L)).thenReturn(xrd);

        RangerSecurityZone zone = new RangerSecurityZone();
        zone.setId(5L);
        Map<String, RangerSecurityZone.RangerSecurityZoneService> services = new HashMap<>();
        RangerSecurityZone.RangerSecurityZoneService zsvc = new RangerSecurityZone.RangerSecurityZoneService();
        HashMap<String, List<String>> res = new HashMap<>();
        res.put("path", Arrays.asList("/a"));
        zsvc.getResources().add(res);
        services.put("s", zsvc);
        zone.setServices(services);
        zone.setAdminUsers(Arrays.asList("u1"));
        zone.setAdminUserGroups(Arrays.asList("g1"));
        zone.setAdminRoles(Arrays.asList("r1"));
        zone.setAuditUsers(Arrays.asList("u2"));
        zone.setAuditUserGroups(Arrays.asList("g2"));
        zone.setAuditRoles(Arrays.asList("r2"));
        zone.setTagServices(Arrays.asList("s"));

        XXUser u1 = new XXUser();
        u1.setId(1L);
        XXGroup g1 = new XXGroup();
        g1.setId(2L);
        XXRole r1 = new XXRole();
        r1.setId(3L);
        XXUserDao xUserDao = mock(XXUserDao.class);
        XXGroupDao xGroupDao = mock(XXGroupDao.class);
        XXRoleDao xRoleDao = mock(XXRoleDao.class);
        when(dao.getXXUser()).thenReturn(xUserDao);
        when(dao.getXXGroup()).thenReturn(xGroupDao);
        when(dao.getXXRole()).thenReturn(xRoleDao);
        when(xUserDao.findByUserName("u1")).thenReturn(u1);
        when(xUserDao.findByUserName("u2")).thenReturn(u1);
        when(xGroupDao.findByGroupName("g1")).thenReturn(g1);
        when(xGroupDao.findByGroupName("g2")).thenReturn(g1);
        when(xRoleDao.findByRoleName("r1")).thenReturn(r1);
        when(xRoleDao.findByRoleName("r2")).thenReturn(r1);

        up.createNewZoneMappingForRefTable(zone);
        verify(dao.getXXSecurityZoneRefService(), times(1)).create(any(XXSecurityZoneRefService.class));
        verify(dao.getXXSecurityZoneRefResource(), times(1)).create(any(XXSecurityZoneRefResource.class));
        verify(dao.getXXSecurityZoneRefTagService(), times(1)).create(any(XXSecurityZoneRefTagService.class));
        verify(dao.getXXSecurityZoneRefUser(), times(2)).create(any(XXSecurityZoneRefUser.class));
        verify(dao.getXXSecurityZoneRefGroup(), times(2)).create(any(XXSecurityZoneRefGroup.class));
        verify(dao.getXXSecurityZoneRefRole(), times(2)).create(any(XXSecurityZoneRefRole.class));
    }

    @Test
    public void test03_updateResourceSignatureWithZoneName_updatesPolicies() throws Exception {
        SecurityZoneRefUpdater up = new SecurityZoneRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        RangerPolicyService policyService = mock(RangerPolicyService.class);
        ServiceDBStore svcStore = mock(ServiceDBStore.class);
        setField(up, SecurityZoneRefUpdater.class, "daoMgr", dao);
        setField(up, SecurityZoneRefUpdater.class, "policyService", policyService);
        setField(up, SecurityZoneRefUpdater.class, "svcStore", svcStore);

        XXPolicy xp = new XXPolicy();
        xp.setId(1L);
        XXPolicyDao xPolicyDao = mock(XXPolicyDao.class);
        when(dao.getXXPolicy()).thenReturn(xPolicyDao);
        when(xPolicyDao.findByZoneId(9L)).thenReturn(Collections.singletonList(xp));
        RangerPolicy rp = new RangerPolicy();
        rp.setId(1L);
        when(policyService.getPopulatedViewObject(xp)).thenReturn(rp);
        RangerSecurityZone z = new RangerSecurityZone();
        z.setId(9L);
        up.updateResourceSignatureWithZoneName(z);
        verify(svcStore, times(1)).updatePolicySignature(any(RangerPolicy.class));
        verify(policyService, times(1)).update(any(RangerPolicy.class));
    }

    @Test
    public void test04_createNewZoneMapping_invalidTagService_throws() throws Exception {
        SecurityZoneRefUpdater up = new SecurityZoneRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        RangerServiceService svcService = mock(RangerServiceService.class);
        RESTErrorUtil rest = mock(RESTErrorUtil.class);
        setField(up, SecurityZoneRefUpdater.class, "daoMgr", dao);
        setField(up, SecurityZoneRefUpdater.class, "svcService", svcService);
        setField(up, SecurityZoneRefUpdater.class, "restErrorUtil", rest);

        // Mocks for cleanupRefTables and service lookup
        XXSecurityZoneRefServiceDao sd = mock(XXSecurityZoneRefServiceDao.class);
        XXSecurityZoneRefTagServiceDao td = mock(XXSecurityZoneRefTagServiceDao.class);
        XXSecurityZoneRefResourceDao rd = mock(XXSecurityZoneRefResourceDao.class);
        XXSecurityZoneRefUserDao ud = mock(XXSecurityZoneRefUserDao.class);
        XXSecurityZoneRefGroupDao gd = mock(XXSecurityZoneRefGroupDao.class);
        XXSecurityZoneRefRoleDao rld = mock(XXSecurityZoneRefRoleDao.class);
        when(dao.getXXSecurityZoneRefService()).thenReturn(sd);
        when(dao.getXXSecurityZoneRefTagService()).thenReturn(td);
        when(dao.getXXSecurityZoneRefResource()).thenReturn(rd);
        when(dao.getXXSecurityZoneRefUser()).thenReturn(ud);
        when(dao.getXXSecurityZoneRefGroup()).thenReturn(gd);
        when(dao.getXXSecurityZoneRefRole()).thenReturn(rld);
        when(sd.findByZoneId(5L)).thenReturn(Collections.emptyList());
        when(td.findByZoneId(5L)).thenReturn(Collections.emptyList());
        when(rd.findByZoneId(5L)).thenReturn(Collections.emptyList());
        when(ud.findByZoneId(5L)).thenReturn(Collections.emptyList());
        when(gd.findByZoneId(5L)).thenReturn(Collections.emptyList());
        when(rld.findByZoneId(5L)).thenReturn(Collections.emptyList());

        XXServiceDao xServiceDao = mock(XXServiceDao.class);
        when(dao.getXXService()).thenReturn(xServiceDao);
        when(xServiceDao.findByName("bad")).thenReturn(null);

        RangerSecurityZone zone = new RangerSecurityZone();
        zone.setId(5L);
        zone.setServices(Collections.emptyMap());
        zone.setAdminUsers(Collections.emptyList());
        zone.setAdminUserGroups(Collections.emptyList());
        zone.setAdminRoles(Collections.emptyList());
        zone.setAuditUsers(Collections.emptyList());
        zone.setAuditUserGroups(Collections.emptyList());
        zone.setAuditRoles(Collections.emptyList());
        zone.setTagServices(Arrays.asList("bad"));

        RuntimeException expected = new RuntimeException("invalid");
        when(rest.createRESTException(any(String.class), any(MessageEnums.class))).thenThrow(expected);
        Assertions.assertThrows(RuntimeException.class, () -> up.createNewZoneMappingForRefTable(zone));
    }
}
