/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql.catalyst.analysis.resolver

import java.util.IdentityHashMap

import scala.collection.mutable

import org.apache.spark.SparkException
import org.apache.spark.sql.catalyst.expressions.{Alias, Attribute, Expression, NamedExpression}
import org.apache.spark.sql.catalyst.expressions.aggregate.AggregateExpression
import org.apache.spark.sql.catalyst.plans.logical.Aggregate

/**
 * Used to extract aggregate and grouping expressions in the context of underlying [[Aggregate]]
 * operator or collecting aggregate expressions based on the provided expression.
 */
class GroupingAndAggregateExpressionsExtractor(
    aggregate: Aggregate,
    autoGeneratedAliasProvider: AutoGeneratedAliasProvider) {

  /**
   * Maps children of aliases from aggregate list to their parents or to `None` if the expression
   * doesn't have an alias. This map only accounts for the first appearance of the expression. For
   * example, for `SELECT col1, col1 AS a` map entry should be (col1 -> None), but for
   * `SELECT col1 AS a, col1` map entry should be (col1 -> Some(a)).
   */
  private val aliasChildToAliasInAggregateExpressions =
    new IdentityHashMap[Expression, Option[Alias]]
  private val aggregateExpressionsSemanticComparator = new SemanticComparator(
    aggregate.aggregateExpressions.map {
      case alias: Alias =>
        aliasChildToAliasInAggregateExpressions.putIfAbsent(alias.child, Some(alias))
        alias.child
      case other =>
        aliasChildToAliasInAggregateExpressions.put(other, None)
        other
    }
  )

  private val groupingExpressionsSemanticComparator = new SemanticComparator(
    aggregate.groupingExpressions
  )

  /**
   * Extracts grouping and aggregate expressions based on the provided expression. Instead of
   * provided expression use the extracted one (if any), update `referencedGroupingExpressions`
   * (grouping expressions used for extraction) and `extractedAggregateExpressionAliases` (aliases
   * of [[AggregateExpression]]s that are transformed to attributes during extraction) in order to
   * insert missing attributes to below operators.
   *
   * When an expression exists in both grouping and aggregate expressions (for example, when there
   * are lateral column references in [[Aggregate]], LCA algorithm will copy grouping expressions
   * to aggregate list prior to entering this method), we still extract grouping expression but
   * don't add it later if it is not necessary.
   */
  def extractReferencedGroupingAndAggregateExpressions(
      expression: Expression,
      referencedGroupingExpressions: mutable.ArrayBuffer[NamedExpression],
      extractedAggregateExpressionAliases: mutable.ArrayBuffer[Alias]): Expression = {
    val aggregateExpressionWithAlias = collectFirstAggregateExpression(expression)
    val isGroupingExpression = groupingExpressionsSemanticComparator.exists(expression)
    if (isGroupingExpression) {
      val groupingExpressionReference = aggregateExpressionWithAlias match {
        case (Some(attribute: Attribute), None) => attribute
        case (Some(_), Some(alias)) => alias.toAttribute
        case _ =>
          expression match {
            case attribute: Attribute => attribute
            case other => autoGeneratedAliasProvider.newAlias(child = other)
          }
      }
      referencedGroupingExpressions += groupingExpressionReference
      groupingExpressionReference.toAttribute
    } else {
      aggregateExpressionWithAlias match {
        case (Some(attribute: Attribute), None) => attribute
        case (Some(_), Some(alias)) => alias.toAttribute
        case (Some(expression), None) =>
          throw SparkException.internalError(
            s"No parent alias for expression $expression while extracting aggregate" +
            s"expressions in an operator."
          )
        case _ =>
          expression match {
            case aggregateExpression: AggregateExpression =>
              val alias = autoGeneratedAliasProvider.newAlias(child = aggregateExpression)
              extractedAggregateExpressionAliases += alias
              alias.toAttribute
            case other => other
          }
      }
    }
  }

  /**
   * Get first aggregate expression's child that is semantically equal to the provided expression,
   * and it's parent [[Alias]] (if any).
   */
  def collectFirstAggregateExpression(
      expression: Expression): (Option[Expression], Option[Alias]) = {
    val referencedAggregateExpression =
      aggregateExpressionsSemanticComparator.collectFirst(expression)
    referencedAggregateExpression match {
      case Some(expression) =>
        (Some(expression), aliasChildToAliasInAggregateExpressions.get(expression))
      case None =>
        (None, None)
    }
  }

  /**
   * Returns true if the provided expression is semantically equal to any of the aggregate
   * expressions' child.
   */
  def exists(expression: Expression): Boolean = {
    aggregateExpressionsSemanticComparator.exists(expression)
  }
}
